/*
 * max7360_keypad.c - MAX7360 Key Switch Controller Driver.
 *
 * Copyright (C) 2009 Samsung Electronics
 * Kim Kyuwon <q1.kim@samsung.com>
 *
 * Based on pxa27x_keypad.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * Datasheet: http://www.maxim-ic.com/quick_view2.cfm/qv_pk/5456
 */



#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/input.h>
#include <linux/input/matrix_keypad.h>
#include <linux/delay.h>
#include <linux/signal.h>
#include <asm/siginfo.h>	//siginfo

#include "max7360.h"


#include <linux/proc_fs.h>  /* Necessary because we use proc fs */
#include <asm/uaccess.h>    /* for copy_*_user */

#include <linux/crestron.h>


#define HARDKEY_FILENAME           "keyBtns"
#define BACKLIGHT_FILENAME           "keyBtnsBkLt"
#define PROCFS_MAX_SIZE     1024
#define NUMBER_OF_HARDKEYS 17

#if 0 
#define KEY_BUTTON_SIGNAL		44
#endif

/* Backlight interface parameters */
#define NUM_OF_LIGHTS 1
#define NUMBER_OF_COLORS 1 	//0 = Off No backlight, 
				//1 = first color, 
				//2 = alt color 1....
#define DEFAULT_COLOR 1
#define MIN_BRIGHTNESS 0
#define MAX_BRIGHTNESS 100 // UIHAL max brightness = 100%
#define DEFAULT_BRIGHTNESS 10

#define MIN_DEV_BRIGHTNESS 255 //actual device MIN value, MAX=0, 


#define MAX7360_MAX_KEY_ROWS	8
#define MAX7360_USED_KEY_ROWS	3
#define MAX7360_MAX_KEY_COLS	8
#define MAX7360_USED_KEY_COLS	5
#define MAX7360_MAX_KEY_NUM	(MAX7360_MAX_KEY_ROWS * MAX7360_MAX_KEY_COLS)
#define MAX7360_ROW_SHIFT	3

/*
 * MAX7360 registers
 */
#define MAX7360_REG_KEYFIFO	0x00
#define MAX7360_REG_CONFIG	0x01
#define MAX7360_REG_DEBOUNCE	0x02
#define MAX7360_REG_INTERRUPT	0x03
#define MAX7360_REG_PORTS	0x04
#define MAX7360_REG_KEYREP	0x05
#define MAX7360_REG_SLEEP	0x06

/*
 * Configuration register bits
 */
#define MAX7360_CFG_SLEEP	(1 << 7)
#define MAX7360_CFG_INTERRUPT	(1 << 5)
#define MAX7360_CFG_KEY_RELEASE	(1 << 3)
#define MAX7360_CFG_WAKEUP	(1 << 1)
#define MAX7360_CFG_TIMEOUT	(1 << 0)

/*
 * Autosleep register values (ms)
 */
#define MAX7360_AUTOSLEEP_8192	0x01
#define MAX7360_AUTOSLEEP_4096	0x02
#define MAX7360_AUTOSLEEP_2048	0x03
#define MAX7360_AUTOSLEEP_1024	0x04
#define MAX7360_AUTOSLEEP_512	0x05
#define MAX7360_AUTOSLEEP_256	0x06

/*
 * KEYPAD LED defines
*/
#define MAX7360_REG_GPIO_GCFG   (0x40)  // R/W | GPIO global configuration.
#define MAX7360_REG_GPIO_CTL    (0x41)  // R/W | GPIO control.
#define MAX7360_REG_GPIO_DEB    (0x42)  // R/W | GPIO debounce.
#define MAX7360_REG_GPIO_CC     (0x43)  // R/W | GPIO constant-current setting.
#define MAX7360_REG_GPIO_OUT    (0x44)  // R/W | GPIO output mode.
#define MAX7360_REG_CMN_PWM     (0x45)  // R/W | Common PWM.
#define MAX7360_REG_ROT_CFG     (0x46)  // R/W | Rotary switch configuration.
#define MAX7360_REG_I2C_TO      (0x48)  //  R  | I2C timeout flag.
#define MAX7360_REG_GPIO_IN     (0x49)  //  R  | GPIO input register.
#define MAX7360_REG_ROT_CNT     (0x4A)  //  R  | Rotary switch count.
#define MAX7360_REG_PORT0_PWM	(0x50)
#define MAX7360_REG_PORT1_PWM	(0x51)
#define MAX7360_REG_PORT2_PWM   (0x52)  // R/W | PORT2 PWM.
#define MAX7360_REG_PORT3_PWM   (0x53)  // R/W | PORT3 PWM.
#define MAX7360_REG_PORT4_PWM   (0x54)  // R/W | PORT4 PWM.
#define MAX7360_REG_PORT5_PWM   (0x55)  // R/W | PORT5 PWM.
#define MAX7360_REG_PORT6_PWM   (0x56)  // R/W | PORT6 PWM.
#define MAX7360_REG_PORT7_PWM   (0x57)  // R/W | PORT7 PWM.
#define MAX7360_REG_PORT0_CFG   (0x58)  // R/W | PORT0 configuration.
#define MAX7360_REG_PORT1_CFG   (0x59)  // R/W | PORT1 configuration.
#define MAX7360_REG_PORT2_CFG   (0x5A)  // R/W | PORT2 configuration.
#define MAX7360_REG_PORT3_CFG   (0x5B)  // R/W | PORT3 configuration.
#define MAX7360_REG_PORT4_CFG   (0x5C)  // R/W | PORT4 configuration.
#define MAX7360_REG_PORT5_CFG   (0x5D)  // R/W | PORT5 configuration.
#define MAX7360_REG_PORT6_CFG   (0x5E)  // R/W | PORT6 configuration.
#define MAX7360_REG_PORT7_CFG   (0x5F)  // R/W | PORT7 configuration.

struct max7360_keypad {
        /* matrix key code map */
        unsigned short keycodes[MAX7360_MAX_KEY_NUM];

        struct input_dev *input_dev;
        struct i2c_client *client;
};

static int max7360_controlBrightness(struct i2c_client *client, int rightness);
static int max7360_controlColor(struct i2c_client *client, int color);
static int max7360_controlID(struct i2c_client *client, int id);


static unsigned int gKeyStates = 0;
static int gBrightness = DEFAULT_BRIGHTNESS;
static int gBrightnessDef = DEFAULT_BRIGHTNESS;
static int gColor = DEFAULT_COLOR;
static int gID = 1;
static struct max7360_keypad *gKeypad = NULL;

#if 0
static unsigned int gSignalPID = 0;
#endif

/** * The structure keeping information about the /proc file */
static struct proc_dir_entry *HardKey_File;
static struct proc_dir_entry *BackLight_File;

/** * The buffer (2k) for this module */
static char gHardKey_buffer[PROCFS_MAX_SIZE];
static char gBackLight_buffer[PROCFS_MAX_SIZE];

/** * The size of the data held in the buffer */
static unsigned long gHardKey_buffer_size = 0;
static unsigned long gBackLight_buffer_size = 0;

#define KEY_DEBUG_FS 1
#ifdef KEY_DEBUG_FS
static int g_dbg_keys=0;
#endif

/** * This funtion is called when the /proc hardkey file is read */
static ssize_t hardKey_read(struct file *filp,	/* see include/linux/fs.h   */
			     char *buffer,	/* buffer to fill with data */
			     size_t length,	/* length of the buffer     */
			     loff_t * offset)
{
  static int finished = 0;
  char * p = gHardKey_buffer;
  int value = gKeyStates;

  /* needed to stop from continuously printing */ 
  if ( finished == 1 ) { finished=0; return 0; }
  finished = 1;

   p += sprintf ( p, "KEYVALUES=%08X\n" , value );
   p += sprintf ( p, "NUMBER_OF_HARDKEYS=%04d\n" , NUMBER_OF_HARDKEYS );

   gHardKey_buffer_size = p-gHardKey_buffer;

	/*
	 * We use put_to_user to copy the string from the kernel's
	 * memory segment to the memory segment of the process
	 * that called us. get_from_user, BTW, is
	 * used for the reverse.
	 */
	if ( copy_to_user(buffer, gHardKey_buffer, gHardKey_buffer_size) ) {
		return -EFAULT;
	}

  return gHardKey_buffer_size;
}

static ssize_t hardKey_write(struct file *file, const char *buffer, size_t len, loff_t * off)
{
	/* no Write support */
	printk(KERN_ERR "Write not supported on this device, only for debug\n");

	/* code below is for debug only  it will override defined button presses 
		and enable some basic Android buttons like back and menu
	*/
        char* tag = NULL;
        char* value = NULL;
        char** tempPtr = &buffer;

	gHardKey_buffer_size = len;
        if (gHardKey_buffer_size > PROCFS_MAX_SIZE ) {
                gHardKey_buffer_size = PROCFS_MAX_SIZE;
        }

        if ( copy_from_user(gHardKey_buffer, buffer, gHardKey_buffer_size) )
        {
                return -EFAULT;
        }
        tag = strsep ( tempPtr, "=" );

        if ( strcmp ( tag, "DEBUG" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &g_dbg_keys );
        }

	return gHardKey_buffer_size;
}

/** * This funtion is called when the /proc brightness file is read */
static ssize_t backLight_read(struct file *filp,  /* see include/linux/fs.h   */
                             char *buffer,      /* buffer to fill with data */
                             size_t length,     /* length of the buffer     */
                             loff_t * offset)
{
  static int finished = 0;
  char * p = gBackLight_buffer;

  /* needed to stop from continuously printing */ 
  if ( finished == 1 ) { finished=0; return 0; }
  finished = 1;

   p += sprintf ( p, "NUM_OF_LIGHTS=%04X\n" , NUM_OF_LIGHTS );
   p += sprintf ( p, "NUMBER_OF_COLORS=%04d\n", NUMBER_OF_COLORS);
   p += sprintf ( p, "DEFAULT_COLOR=%04d\n", DEFAULT_COLOR );
   p += sprintf ( p, "MIN_BRIGHTNESS=%04d\n" ,MIN_BRIGHTNESS);
   p += sprintf ( p, "MAX_BRIGHTNESS=%04d\n" ,MAX_BRIGHTNESS);
   p += sprintf ( p, "DEFAULT_BRIGHTNESS=%04d\n", gBrightnessDef);
   p += sprintf ( p, "BRIGHTNESS=%04d\n", gBrightness );
   p += sprintf ( p, "COLOR=%04d\n", gColor );

   gBackLight_buffer_size = p-gBackLight_buffer;

        /*
         * We use put_to_user to copy the string from the kernel's
         * memory segment to the memory segment of the process
         * that called us. get_from_user, BTW, is
         * used for the reverse.
         */
        if ( copy_to_user(buffer, gBackLight_buffer, gBackLight_buffer_size) ) {
                return -EFAULT;
        }

  return gBackLight_buffer_size;
}


static ssize_t backLight_write(struct file *file, const char *buffer, size_t len, loff_t * off)
{
  	char* tag = NULL;
  	char* value = NULL;
  	char** tempPtr = &buffer;

        /* get buffer size */
        gBackLight_buffer_size = len;
        if (gBackLight_buffer_size > PROCFS_MAX_SIZE ) {
                gBackLight_buffer_size = PROCFS_MAX_SIZE;
        }

        /* write data to the buffer */
        if ( copy_from_user(gBackLight_buffer, buffer, gBackLight_buffer_size) )
        {
                return -EFAULT;
        }
        tag = strsep ( tempPtr, "=" );

        if ( strcmp ( tag, "BRIGHTNESS" ) == 0 )
        {
          struct max7360_keypad *keypad = gKeypad;
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &gBrightness );
          if ( gKeypad != NULL )
          {
            max7360_controlBrightness(keypad->client, gBrightness );
          }
        }
        if ( strcmp ( tag, "COLOR" ) == 0 )
        {
          struct max7360_keypad *keypad = gKeypad;
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &gColor );
          if ( gKeypad != NULL )
          {
            max7360_controlColor(keypad->client, gColor );
          }
        }
        if ( strcmp ( tag, "ID" ) == 0 )
        {
          struct max7360_keypad *keypad = gKeypad;
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &gColor );
          if ( gKeypad != NULL )
          {
            max7360_controlID(keypad->client, gID );
          }
        }

        return gBackLight_buffer_size;
}

/*
 * This function decides whether to allow an operation
 * (return zero) or not allow it (return a non-zero
 * which indicates why it is not allowed).
 *
 * The operation can be one of the following values:
 * 0 - Execute (run the "file" - meaningless in our case)
 * 2 - Write (input to the kernel module)
 * 4 - Read (output from the kernel module)
 *
 * This is the real function that checks file
 * permissions. The permissions returned by ls -l are
 * for referece only, and can be overridden here.
 */

static int module_permission(struct inode *inode, int op, struct nameidata *foo)
{
//  if ( op == 2 ) // no writes
//  {
//    return -EACCES;
//  }

  return 0;
}

/*
 * The file is opened - we don't really care about
 * that, but it does mean we need to increment the
 * module's reference count.
 */
int hardKey_open_procfs(struct inode *inode, struct file *file)
{
	try_module_get(THIS_MODULE);
	return 0;
}

/*
 * The file is closed - again, interesting only because
 * of the reference count.
 */
int hardKey_close_procfs(struct inode *inode, struct file *file)
{
	module_put(THIS_MODULE);
	return 0;		/* success */
}

static struct file_operations File_Ops_HardKey_File = {
	.read 	 = hardKey_read,
	.write 	 = hardKey_write,
	.open 	 = hardKey_open_procfs,
	.release = hardKey_close_procfs,
};

/*
 * Inode operations for our proc file. We need it so
 * we'll have some place to specify the file operations
 * structure we want to use, and the function we use for
 * permissions. It's also possible to specify functions
 * to be called for anything else which could be done to
 * an inode (although we don't bother, we just put
 * NULL).
 */

static struct inode_operations Inode_Ops_File = {
	.permission = module_permission,	/* check for permissions */
};

/*
 * The file is opened - we don't really care about
 * that, but it does mean we need to increment the
 * module's reference count.
 */
int backLight_open_procfs(struct inode *inode, struct file *file)
{
        try_module_get(THIS_MODULE);
        return 0;
}

/*
 * The file is closed - again, interesting only because
 * of the reference count.
 */
int backLight_close_procfs(struct inode *inode, struct file *file)
{
        module_put(THIS_MODULE);
        return 0;               /* success */
}

static struct file_operations File_Ops_BackLight_File = {
        .read    = backLight_read,
        .write   = backLight_write,
        .open    = backLight_open_procfs,
        .release = backLight_close_procfs,
};

static int max7360_write_reg(struct i2c_client *client, u8 reg, u8 val)
{
	int ret = i2c_smbus_write_byte_data(client, reg, val);

	if (ret < 0)
		dev_err(&client->dev, "%s: reg 0x%x, val 0x%x, err %d\n",
			__func__, reg, val, ret);
	return ret;
}

static int max7360_read_reg(struct i2c_client *client, int reg)
{
	int ret = i2c_smbus_read_byte_data(client, reg);

	if (ret < 0)
		dev_err(&client->dev, "%s: reg 0x%x, err %d\n",
			__func__, reg, ret);
	return ret;
}

static void max7360_build_keycode(struct max7360_keypad *keypad,
				const struct matrix_keymap_data *keymap_data)
{
	struct input_dev *input_dev = keypad->input_dev;
	int i;
	for (i = 0; i < keymap_data->keymap_size; i++) {
		unsigned int key = keymap_data->keymap[i];
		unsigned int row = KEY_ROW(key);
		unsigned int col = KEY_COL(key);
		unsigned int scancode = MATRIX_SCAN_CODE(row, col,
						MAX7360_ROW_SHIFT);
		unsigned short keycode = KEY_VAL(key);
		keypad->keycodes[scancode] = keycode;
//printk(KERN_ERR "RJK:: row= %d col=%d key=%d\n", row, col, key);
//printk(KERN_ERR "RJK:: keycode=%d\n", keycode);
		__set_bit(keycode, input_dev->keybit);
	}
	__clear_bit(KEY_RESERVED, input_dev->keybit);
}

/* convert 100% to 255 - 1 values; device levels: 255 is dim 0 is brightest
 * 0 = Max brightness
 * 255 = minimum brightness
*/
static int max7360_controlBrightness(struct i2c_client *client, int brightness )
{
  	
  u32 val;

  if (brightness) {
	if (brightness==100)
		val=0;  //MAX brightness
	else
  		val = 255 - (((brightness*10) * (MIN_DEV_BRIGHTNESS/10)) / 100); //max is 100% converted to 250
  }
  else
	val=255; //if brightneess is zero set display to min value(255)

//printk(KERN_ERR " RJK: in val=%d out val=%d\n", brightness, val);

  max7360_write_reg(client, MAX7360_REG_PORT1_PWM,  val);

  return 0;
}
static int max7360_controlColor(struct i2c_client *client, int color )
{
  int val = color;
  printk(KERN_ERR "tst600 COLOR not supported, (color=1)\n");
  //max7360_write_reg(client, MAX7360_REG_PORT1_PWM,  val);
  return 0;
}
static int max7360_controlID(struct i2c_client *client, int id )
{
  int val = id;
  printk(KERN_ERR "tst600 ID not supported. (ID=1)\n");
  //max7360_write_reg(client, MAX7360_REG_PORT1_PWM,  val);
  return 0;
}

static int max7360_toggleLed(struct i2c_client *client)
{
	int val;

	printk(KERN_ERR "toggle KeyPadLed\n");
	val=max7360_read_reg(client, MAX7360_REG_PORT1_PWM);
	/* brightness range is 0xff - 0x00; 0xFF=off  */
	if (val == 0x00 )
		max7360_write_reg(client, MAX7360_REG_PORT1_PWM,  0x30);
	if (val == 0x30 )
		max7360_write_reg(client, MAX7360_REG_PORT1_PWM,  0xD0);
	if (val == 0xD0 )
		max7360_write_reg(client, MAX7360_REG_PORT1_PWM,  0xFF);
	if (val == 0xFF )
		max7360_write_reg(client, MAX7360_REG_PORT1_PWM,  0x00);
	return 0;
}


/* runs in an IRQ thread -- can (and will!) sleep */
static irqreturn_t max7360_interrupt(int irq, void *dev_id)
{
	struct max7360_keypad *keypad = dev_id;
	struct input_dev *input_dev = keypad->input_dev;
	int val, row, col, release, code;
	bool hardwareBtn=false;

	val = max7360_read_reg(keypad->client, MAX7360_REG_KEYFIFO);
	row = val & 0x7;
	col = (val >> 3) & 0x7;
	release = val & 0x40;

	code = MATRIX_SCAN_CODE(row, col, MAX7360_ROW_SHIFT);

//printk(KERN_ERR "RJK:: max7360: key[%d:%d] %s\n", row, col, release ? "release" : "press");
	dev_dbg(&keypad->client->dev,
		"key[%d:%d] %s\n", row, col, release ? "release" : "press");

  if ( g_dbg_keys == 0 )
  {
	  input_report_key(input_dev, keypad->keycodes[code], !release);
	}
	else
	{
    int i = 0;
    for (i = 0; i < ARRAY_SIZE(max7360_matrix_keys); i++)
	  {
       unsigned int key = KEY_VAL(max7360_matrix_keysDEBUG[i]);
	     if ( KEY_VAL( max7360_matrix_keys[i] ) == keypad->keycodes[code] )
       {
	     	 if ( key == KEY_6 )
	     	 {
	     		 max7360_toggleLed(keypad->client);
	     		input_report_key(input_dev, keypad->keycodes[code], !release);
	     	 }
	     	 else
	     	 {
	         input_report_key(input_dev, key, !release);
	         __set_bit(key, input_dev->keybit);
      		 __clear_bit(KEY_RESERVED, input_dev->keybit);
	         input_report_key(input_dev, key, !release);
	         input_report_key(input_dev, keypad->keycodes[code], !release);
	     	 }
       }
    }
	}
  input_sync(input_dev);

	/* Keyevents are set in bitmask for reading by Crestron UIHal layer */
        if ( release != 0 )
        {
          gKeyStates &= ~( 0x0001 << ( ( col * MAX7360_USED_KEY_ROWS ) + ( row ) ) );
        }
        else
        {
          gKeyStates |= ( 0x0001 << ( ( col * MAX7360_USED_KEY_ROWS ) + ( row ) ) );
        }
#if 0  //Signaling stuff, decided not to use this
        if ( gSignalPID > 0 )
        {
          // send signal
          int res;
	  struct siginfo info;

	  memset(&info, 0, sizeof(struct siginfo));
	  info.si_signo = KEY_BUTTON_SIGNAL;
	  info.si_code = SI_QUEUE;
	  info.si_int = gKeyStates;  

	  res = kill_proc_info(KEY_BUTTON_SIGNAL, &info, gSignalPID);//send signal to user land 	  if (ret < 0) 
          if ( res != 0 )
          {
	    printk("error sending signal\n");
            gSignalPID = 0;
	  }
        }
#endif
	return IRQ_HANDLED;
}

/*
 * Let MAX7360 fall into a deep sleep:
 * If no keys are pressed, enter sleep mode for 8192 ms. And if any
 * key is pressed, the MAX7360 returns to normal operating mode.
 */
static inline void max7360_fall_deepsleep(struct i2c_client *client)
{
	max7360_write_reg(client, MAX7360_REG_SLEEP, MAX7360_AUTOSLEEP_8192);
}

/*
 * Let MAX7360 take a catnap:
 * Autosleep just for 256 ms.
 */
static inline void max7360_take_catnap(struct i2c_client *client)
{
	max7360_write_reg(client, MAX7360_REG_SLEEP, MAX7360_AUTOSLEEP_256);
}

static int max7360_open(struct input_dev *dev)
{
	struct max7360_keypad *keypad = input_get_drvdata(dev);

	max7360_take_catnap(keypad->client);

	return 0;
}

static void max7360_close(struct input_dev *dev)
{
	struct max7360_keypad *keypad = input_get_drvdata(dev);

	max7360_fall_deepsleep(keypad->client);
}

static int max7360_initialize(struct i2c_client *client)
{
#if 0    //my config
	max7360_write_reg(client, MAX7360_REG_CONFIG,
		MAX7360_CFG_INTERRUPT | /* Irq clears after host read */
		MAX7360_CFG_KEY_RELEASE | /* Key release enable */
		MAX7360_CFG_WAKEUP); /* Key press wakeup enable */
#endif
#if 1  //This is winCE config 
	max7360_write_reg(client, MAX7360_REG_CONFIG,
		MAX7360_CFG_SLEEP | /* sleep enable */
		MAX7360_CFG_KEY_RELEASE | /* Key release enable */
		MAX7360_CFG_WAKEUP); /* Key press wakeup enable */
#endif

	/* Full key-scan functionality */
	//max7360_write_reg(client, MAX7360_REG_DEBOUNCE, 0x1F);
	max7360_write_reg(client, MAX7360_REG_DEBOUNCE, 0x7F);  //WinCE value

	/* nINTK asserts every debounce cycles */
	max7360_write_reg(client, MAX7360_REG_INTERRUPT, 0x01);

	/* PORT nINTK GPO control  */
	max7360_write_reg(client, MAX7360_REG_PORTS, 0xfc);

    	// Disable MAX7360 chip's "autosleep" feature:  Instead of automatically
    	//  going to sleep on its own, whenever no keys have been pressed for a
    	//  short time, the chip should follow the system's power-saving mode.
    	//
    	// NOTE:  Supporting "autosleep" would cause a problem, because it would
    	//  require another device driver to behave in an unexpected way.  The
    	//  MAX7360 chip handles both the keypad (current device driver) and the
    	//  keypad backlight LEDs (another device driver).  When the chip enters
    	//  sleep mode, it shuts-down the PWM/output pins that control the LEDs,
    	//  and they go dark.  Such behavior is desirable when the whole system
    	//  is suspended/hibernating, but not while it is in normal operation.
    	//  Therefore, "autosleep" is disabled, so the keypad backlight LEDs can
    	//  remain lit during normal operation.
    	//
    	// - Reference:  Maxim Integrated Products MAX7360 datasheet, Rev. 0,
    	//   dated "4/09" (Note relationship between Configuration register, bit
    	//   "D7"; and GPIO Global Configuration register, bit "D4").
    	//
	//max7360_fall_deepsleep(client);
 	max7360_write_reg(client, MAX7360_REG_SLEEP, 0);

	/* init keypad LED driver */
        // First step of initialization, in order get a "clean start":
        //  Reset all GPIO registers to default/power-on-reset values.
        //
        //KBL_WriteRegister(KBL_MAX_REG_GPIO_GCFG, 0x08, KBL_KEYPAD_RIGHT_BANK) &&
 	max7360_write_reg(client, MAX7360_REG_GPIO_GCFG, 0x08);
        //- - - - -
        // Configure all GPIO pins ('PORTx'):
        //  - There is no rotary switch.
        //  - Interrupt pin '~INTI' is not connected.
        //  - GPIO/PWM signals are enabled.
        //  - Normal operation.
        //  - Not using PWM/LED fading feature.
        //
        //KBL_WriteRegister(KBL_MAX_REG_GPIO_GCFG, KBL_MAX_REG_GPIO_GCFG_POWER_UP, KBL_KEYPAD_RIGHT_BANK) &&
 	max7360_write_reg(client, MAX7360_REG_GPIO_GCFG, 0x10);
        //- - - - -
        // Allow time to transition from sleep mode to normal operation.
        //  - Sleep/shutdown mode is the power-on default.
        //  - According to the MAX7360 datasheet, the "Startup Time from
        //    Shutdown" is up to 2.4 ms, and typically 2 ms.
        //
        //(Sleep(3), true) &&
	msleep(3);
        //- - - - -
        // Configure all GPIO pins ('PORTx') as outputs:
        //  - Pins PORT0 and PORT1 are outputs that control LEDs.
        //  - Pins PORT2-PORT7 are not connected, but "ports consume
        //    additional current if their inputs are left undriven",
        //    according to the MAX7360 datasheet.  Therefore, configure
        //    them as outputs, in order to avoid drawing extra current.
        //
        //KBL_WriteRegister(KBL_MAX_REG_GPIO_CTL, 0xFF, KBL_KEYPAD_RIGHT_BANK) &&
 	max7360_write_reg(client, MAX7360_REG_GPIO_CTL, 0xFF);
        //- - - - -
        // Leave register 'KBL_MAX_REG_GPIO_DEB' at default (0x00 = 9 ms).
        //  - All GPIO pins ('PORTx') are treated as outputs, not inputs.
        //    Therefore, debounce time does not apply.
        //- - - - -
        // Leave register 'KBL_MAX_REG_GPIO_CC' at default (0xC0 = 5 mA).
        //  - Use lowest constant-current setting, just to be safe.
        //  - As configured elsewhere, pins PORT0 and PORT1 are
        //    non-constant current outputs.
        //  - Pins PORT2-PORT7 are not connected --> Don't care.
        //- - - - -
        // Configure output modes of GPIO pins ('PORTx'):
        //  - Pins PORT0 and PORT1 have been working fine with the LEDs
        //    while in "non-constant-current open-drain output" mode.
        //  - Pins PORT2-PORT7 are not connected --> Don't care.
        //
        //KBL_WriteRegister(KBL_MAX_REG_GPIO_OUT, 0xFF, KBL_KEYPAD_RIGHT_BANK) &&
 	max7360_write_reg(client, MAX7360_REG_GPIO_OUT, 0xFF);
        //- - - - -
        // Leave register 'KBL_MAX_REG_CMN_PWM' at default (0x00 = 0/256).
        //  - As configured elsewhere, using individual PWM intensity
        //    settings, instead of the common PWM setting.
        //- - - - -
        // Leave register 'KBL_MAX_REG_ROT_CFG' at default (0x00).
        //  - As configured elsewhere, there is no rotary switch.
        //- - - - -
        // Leave registers 'KBL_MAX_REG_PORTx_PWM' at default (0x00 = 0/256).
        //  - Another function in this s/w module sets LED brightness by
        //    configuring the individual PWMs of pins PORT0 and PORT1.
        //  - Pins PORT2-PORT7 are not connected --> Don't care.
        //- - - - -
        // Configure all GPIO pins ('PORTx') to mask/ignore interrupts:
        //  - As configured elsewhere, pins PORT0-PORT7 are outputs.
        //  - Interrupt pin '~INTI' only reacts to inputs.
        //  - Interrupt pin '~INTI' is not connected --> Don't use.
        //
        // Also, configure all GPIO pins to use individual PWM intensity
        //  registers and not to blink.
        //
        //KBL_WriteRegister(KBL_MAX_REG_PORT0_CFG, 0x80, KBL_KEYPAD_RIGHT_BANK) &&
        //KBL_WriteRegister(KBL_MAX_REG_PORT1_CFG, 0x80, KBL_KEYPAD_RIGHT_BANK) &&
        //KBL_WriteRegister(KBL_MAX_REG_PORT2_CFG, 0x80, KBL_KEYPAD_RIGHT_BANK) &&
        //KBL_WriteRegister(KBL_MAX_REG_PORT3_CFG, 0x80, KBL_KEYPAD_RIGHT_BANK) &&
        //KBL_WriteRegister(KBL_MAX_REG_PORT4_CFG, 0x80, KBL_KEYPAD_RIGHT_BANK) &&
        //KBL_WriteRegister(KBL_MAX_REG_PORT5_CFG, 0x80, KBL_KEYPAD_RIGHT_BANK) &&
        //KBL_WriteRegister(KBL_MAX_REG_PORT6_CFG, 0x80, KBL_KEYPAD_RIGHT_BANK) &&
        //KBL_WriteRegister(KBL_MAX_REG_PORT7_CFG, 0x80, KBL_KEYPAD_RIGHT_BANK) &&
 	max7360_write_reg(client, MAX7360_REG_PORT0_CFG, 0x80);
 	max7360_write_reg(client, MAX7360_REG_PORT1_CFG, 0x80);
 	max7360_write_reg(client, MAX7360_REG_PORT2_CFG, 0x80);
 	max7360_write_reg(client, MAX7360_REG_PORT3_CFG, 0x80);
 	max7360_write_reg(client, MAX7360_REG_PORT4_CFG, 0x80);
 	max7360_write_reg(client, MAX7360_REG_PORT5_CFG, 0x80);
 	max7360_write_reg(client, MAX7360_REG_PORT6_CFG, 0x80);
 	max7360_write_reg(client, MAX7360_REG_PORT7_CFG, 0x80);
        //- - - - -

	/* initialize Backlight to DEFAULT value 
	 * NOTE: this will need to be eventulaly written to a file for persistance
	 *       As well, file will need to be read at startup
	*/
	if ( gKeypad != NULL )
          {
		printk(KERN_ERR "Setting keypad Brightness to %d(default)\n",gBrightnessDef);
		max7360_controlBrightness(client, gBrightnessDef);
          }

 	/* create the /proc file for HardKey bitmask */
	HardKey_File = create_proc_entry(HARDKEY_FILENAME, 0644, NULL);
	if (HardKey_File == NULL){
		printk(KERN_ALERT "Error: Could not initialize /proc/%s\n",
		       HARDKEY_FILENAME);
		return -ENOMEM;
	}
	else
	{
	  //HardKey_File->owner = THIS_MODULE;
	  HardKey_File->proc_iops = &Inode_Ops_File;
	  HardKey_File->proc_fops = &File_Ops_HardKey_File;
	  HardKey_File->mode = S_IFREG | S_IRUGO | S_IWUSR;
	  HardKey_File->uid = 0;
	  HardKey_File->gid = 0;
	  HardKey_File->size = 80;
       }

 	/* create the /proc file for Brightness bitmask */
	BackLight_File = create_proc_entry(BACKLIGHT_FILENAME, 0644, NULL);
	if (BackLight_File == NULL){
		printk(KERN_ALERT "Error: Could not initialize /proc/%s\n",
		       BACKLIGHT_FILENAME);
		return -ENOMEM;
	}
	else
	{
	  //BackLight_File->owner = THIS_MODULE;
	  BackLight_File->proc_iops = &Inode_Ops_File;
	  BackLight_File->proc_fops = &File_Ops_BackLight_File;
	  BackLight_File->mode = S_IFREG | S_IRUGO | S_IWUSR;
	  BackLight_File->uid = 0;
	  BackLight_File->gid = 0;
	  BackLight_File->size = 80;
       }

	return 0;
}

static int __devinit max7360_probe(struct i2c_client *client,
					const struct i2c_device_id *id)
{
	//const struct matrix_keymap_data *keymap_data = client->dev.platform_data;
	const struct matrix_keymap_data *keymap_data = &max7360_keymap_data;
	struct max7360_keypad *keypad;
	struct input_dev *input_dev;
	int ret;
	int error;

	if (!client->irq) {
		dev_err(&client->dev, "The irq number should not be zero\n");
		return -EINVAL;
	}

	/* Detect MAX7360: The initial Keys FIFO value is '0x3F' */
	ret = max7360_read_reg(client, MAX7360_REG_KEYFIFO);
	if (ret < 0) {
		dev_err(&client->dev, "failed to detect device\n");
		return -ENODEV;
	}

	dev_dbg(&client->dev, "keys FIFO is 0x%02x\n", ret);

	/* RJK: allocate memory for keypad struct */
	keypad = kzalloc(sizeof(struct max7360_keypad), GFP_KERNEL);
	input_dev = input_allocate_device();
	if (!keypad || !input_dev) {
		dev_err(&client->dev, "failed to allocate memory\n");
		error = -ENOMEM;
		goto failed_free_mem;
	}

	keypad->client = client;
	keypad->input_dev = input_dev;

	input_dev->name = client->name;
	input_dev->id.bustype = BUS_I2C;
	input_dev->open = max7360_open;
	input_dev->close = max7360_close;
	input_dev->dev.parent = &client->dev;

	input_dev->evbit[0] = BIT_MASK(EV_KEY) | BIT_MASK(EV_REP);
	input_dev->keycodesize = sizeof(keypad->keycodes[0]);
	input_dev->keycodemax = ARRAY_SIZE(keypad->keycodes);
	input_dev->keycode = keypad->keycodes;

	input_set_capability(input_dev, EV_MSC, MSC_SCAN);
	input_set_drvdata(input_dev, keypad);

	max7360_build_keycode(keypad, keymap_data);

	error = request_threaded_irq(client->irq, NULL, max7360_interrupt,
				     IRQF_TRIGGER_LOW | IRQF_ONESHOT,
				     client->name, keypad);
	if (error) {
		dev_err(&client->dev, "failed to register interrupt\n");
		goto failed_free_mem;
	}

	/* Register the input device */
	error = input_register_device(input_dev);
	if (error) {
		dev_err(&client->dev, "failed to register input device\n");
		goto failed_free_irq;
	}

	i2c_set_clientdata(client, keypad);
	device_init_wakeup(&client->dev, 1);
        gKeypad = keypad;

	/* Initialize MAX7360 */
	error = max7360_initialize(client);
	if (error) {
		/* proc stuff did not init properly already prints from init function */	
	}

	return 0;

failed_free_irq:
	free_irq(client->irq, keypad);
failed_free_mem:
	input_free_device(input_dev);
	kfree(keypad);
	return error;
}

static int __devexit max7360_remove(struct i2c_client *client)
{
	struct max7360_keypad *keypad = i2c_get_clientdata(client);

	free_irq(client->irq, keypad);
	input_unregister_device(keypad->input_dev);
	kfree(keypad);
        gKeypad = NULL;

	return 0;
}

#ifdef CONFIG_PM
static int max7360_suspend(struct i2c_client *client, pm_message_t mesg)
{
	max7360_fall_deepsleep(client);

printk(KERN_ERR " RJK: in max7360_suspend\n");

	if (device_may_wakeup(&client->dev))
		enable_irq_wake(client->irq);

	return 0;
}

static int max7360_resume(struct i2c_client *client)
{
printk(KERN_ERR " RJK: in max7360_resume\n");
	if (device_may_wakeup(&client->dev))
		disable_irq_wake(client->irq);

	/* Restore the default setting */
	max7360_take_catnap(client);

	return 0;
}
#else
#define max7360_suspend	NULL
#define max7360_resume	NULL
#endif


static const struct i2c_device_id max7360_ids[] = {
	{ "max7360", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, max7360_ids);

static struct i2c_driver max7360_i2c_driver = {
	.driver = {
		.name = "max7360",
	},
	.probe		= max7360_probe,
	.remove		= __devexit_p(max7360_remove),
	.suspend	= max7360_suspend,
	.resume		= max7360_resume,
	.id_table	= max7360_ids,
};

static int __init max7360_init(void)
{

	if(gProductId == TST600)
    {
		return i2c_add_driver(&max7360_i2c_driver);
	}
	else 
	{
		return 0;
	}
}
module_init(max7360_init);

static void __exit max7360_exit(void)
{
	i2c_del_driver(&max7360_i2c_driver);
}
module_exit(max7360_exit);

